/*
 * Module:       com.unisys.bis.BISScript.java
 *
 * Description:  BISScript interface
 *
 * Product:      Resource Adapter for BIS
 *
 * Date:         May, 2005
 *
 * Copyright (c) 2005 Unisys Corporation.
 *
 * All rights reserved.
 *
 * UNISYS CONFIDENTIAL
 */

package com.unisys.bis;

import java.io.InputStream;

/**
 * The BISScript interface defines the methods used to execute a BIS script.
 * 
 * @author lamberna
 *
 */
public interface BISScript {
        
    /**
     * Called to clear the script parameter and input dataset information.
     *
     */
    public void clear();
    
    /**
     * Called to release resources referenced by the BISScript instance. 
     *
     */
    public void close() throws BISException;
    
    /**
     * Called to execute a BIS script.
     * 
     * @throws BISException  thrown if an error occurs while executing a script.
     */
    public void execute() throws BISException;
    
    /**
     * After executing a script, this method is called to access the dataset returned by 
     * the script as a CLOB.
     * 
     * @return BISClob instance.
     * 
     * @throws BISException  thrown if an error occurs while accessing the CLOB.
     */
    public BISClob getCLOB() throws BISException;
    
    /**
     * After executing a script, this method is called to access the dataset returned by 
     * the script.
     * 
     * @return BISDataset instance.
     * 
     * @throws BISException - thrown if an error occurs while accessing the dataset.
     */
    public BISDataset getDataset() throws BISException;
    
    /**
     * Called to get the function name associated with the BISScript instance.
     * 
     * @return function name
     * 
     * @throws BISException - thrown if an error occurs while getting the function name.
     */
    public String getFunctionName() throws BISException;
    
    /**
     * After executing a script, this method is called to get the return value.
     * 
     * @return script return value.
     * 
     * @throws BISException  thrown if an error occurs while getting the return value.
     */
    public String getReturnValue() throws BISException;
    
    /**
     * Called to get the script name associated with the BISScript instance.
     * 
     * @return script name.
     * @throws BISException  thrown if an error occurs while getting the script name.
     */
    public String getScriptName() throws BISException;
    
    /**
     * Called to get the value of the designated parameter.
     * 
     * @param parameterIndex - an integer value that identifies a parameter.  
     * The first parameter is 1.
     * 
     * @return parameter value
     * 
     * @throws BISException  thrown if the parameterIndex is not valid.
     */
    public String getString(int parameterIndex) throws BISException;
    
    /**
     * Called to register the designated parameter as an output parameter.
     * 
     * @param parameterIndex  an integer value that identifies a parameter.  
     * The first parameter is 1
     * 
     * @throws BISException  thrown if the parameterIndex is not valid.
     */
    public void registerOutputParameter(int parameterIndex) throws BISException;
    
    /**
     * Called to set the functionName.
     * 
     * @param functionName  the name of the function.
     * 
     * @throws BISException  thrown if an error occurs while setting the function name.
     */
    public void setFunctionName(String functionName) throws BISException;
    
    /**
     * Called to designate the input dataset for the script.
     * 
     * @param datasetName  the name of the dataset.
     * 
     * @throws BISException  thrown if an error occurs while setting the input dataset.
     */
    public void setInputDataset(String datasetName) throws BISException;
    
    /**
     * Called to provide a stream of data as input to a BIS script.
     * 
     * The data is formatted based on the width of the specified drawer name,
     * wrap characters and parse direction.
     * 
     * @param drawerName - a BIS database drawer name
     * @param MIMEType - indicates the type of data
     * @param wrapChars - one or more characters that indicate where line breaks should occur in the data
     * @param direction - indicates if forward (F) or backward (B) parsing is used to format the data
     * @param data - an InputStream instance
     * 
     * @throws BISException  thrown if an error occurs while setting the input stream.
     */
    public void setInputStream(String drawerName,
                               String MIMEType,
                               String wrapChars,
                               char direction,
                               InputStream data) throws BISException;
    
    /**
     * Called to designate the input dataset for the script.
     * 
     * @param dataset  a BISDataset instance.
     * 
     * @throws BISException  thrown if an error occurs while setting the input dataset.
     */
    public void setInputDataset(BISDataset dataset) throws BISException;
    
    /**
     * Called to set the scriptName.
     * 
     * @param scriptName  the name of the script.
     * 
     * @throws BISException  thrown if an error occurs while setting the script name.
     */
    public void setScriptName(String scriptName) throws BISException;
    
    /**
     * Called to set the value of the designated parameter.
     * 
     * @param parameterIndex  an integer value that identifies a parameter.  
     * The first parameter is 1.
     * @param value  the parameter value.
     * 
     * @throws BISException  thrown if the parameterIndex is not valid.
     */
    public void setString(int parameterIndex, String value) throws BISException;
}
