/*
 * Module:       com.unisys.bis.BISResourceAdapter.java
 *
 * Description:  BISResourceAdapter class
 *
 * Product:      Resource Adapter for BIS
 *
 * Date:         May, 2005
 *
 * Copyright (c) 2005 Unisys Corporation.
 *
 * All rights reserved.
 *
 * UNISYS CONFIDENTIAL
 */

package com.unisys.bis;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.resource.ResourceException;
import javax.resource.spi.InvalidPropertyException;
import javax.resource.spi.ResourceAdapterInternalException;

import com.unisys.bis.impl.BISManagedConnectionFactoryImpl;
import com.unisys.bis.impl.BISResourceAdapterImpl;

/**
 * The BISResourceAdapter class is used in acquire a BISConnectionFactory instance 
 * for use in a standalone Java application.
 * 
 * @author lamberna
 *
 */
public class BISResourceAdapter {
    /*
     * Resource Adapter Configuration Parameter IDs.
     */
    
    /**
     * The maximum number of bytes returned by the JBIS Server when accessing a BIS dataset.  
     * This is intended to improve performance when accessing very large datasets.
     * <p>
     * Value Object: java.lang.Integer
     * <p>
     * Default value: 8192 (8 KB) <br> 
     * Minimum Value: 4096 (4 KB) <br>
     * Maximum Value: 800000 (~780 KB) <br>
     * <p>
     * Note: This value does not apply when accessing BLOB datasets.
     * 
     */
    public static final int BIS_BLOCK_SIZE = 1;
    
    /**
     * The character set name used for encoding String values.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: ISO8859-1
     */
    public static final int BIS_CHARACTER_SET = 2;
    
    /**
     * A string that identifies the connection type. Possible values are SOCKETS and SECURE-SOCKETS.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: SOCKETS
     */
    public static final int BIS_CONNECTION_TYPE = 3; 
    
    /**
     * The department number associated with the user name.
     * <p>
     * Value Object: java.lang.String
     */
    public static final int BIS_DEPARTMENT = 4; 
    
    /**
     * Java locale designator. The locale is typically a 2-character or 5-character 
     * string containing a 2-character lowercase language code, optionally followed 
     * by an underscore with a 2-character uppercase country code.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * For example: en_US, de, de_CH.
     */
    public static final int BIS_LOCALE = 5; 
    
    /**
     * The password associated with the user name. 
     * The password may be null.
     * <p>
     * Note: Passwords sent from the Business Information Server Resource Adapter to 
     * the JBIS Server are encrypted to help prevent unauthorized access to a server.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Maximum length: 18
     */
    public static final int BIS_PASSWORD = 6; 
    
    /**
     * The port number used to connect to the JBIS Server.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: 3984
     */
    public static final int BIS_PORT_NUMBER = 7; 
    
    /**
     * The name of the system on which JBIS Server is running.
     * <p>
     * Value Object: java.lang.String
     */
    public static final int BIS_SERVER_NAME = 8;
    
    /**
     * The Business Information Server site to access.
     * <p>
     * Value Object: java.lang.String
     */
    public static final int BIS_SITE = 9; 
    
    /**
     * The user name for signing on to BIS.
     * <p>
     * Value Object: java.lang.String
     */
    public static final int BIS_USER_NAME = 10; 

    /**
     * This static method is called to get a BISConnectionFactory instance when an application 
     * is executed in an unmanaged environment.
     * 
     * @param connectionProperties  a Map instance that contains connection property IDs and values.<br>
     * Property IDs:<br>
     * BIS_BLOCK_SIZE = 1<br>
     * BIS_CHARACTER_SET = 2<br>
     * BIS_CONNECTION_TYPE = 3<br>
     * BIS_DEPARTMENT = 4<br>
     * BIS_LOCALE = 5<br>
     * BIS_PASSWORD = 6<br>
     * BIS_PORT_NUMBER = 7<br>
     * BIS_SERVER_NAME = 8<br>
     * BIS_SITE = 9<br>
     * BIS_USER_NAME = 10
     * 
     * @return BISConnectionFactory instance.
     * 
     * @throws BISException  thrown if an error occurs while creating a BISConnectionFactory.
     */
    public static BISConnectionFactory createConnectionFactory( Map connectionProperties )
                                                  throws BISException {
        /*
         * The implementation will construct a BISResourceAdapterImpl instance and a 
         * BISManagedConnectionFactoryImpl instance.  The BISResourceAdapterImpl.start() method is 
         * called to execute startup logic.
         */
        BISResourceAdapterImpl resourceAdapter = new BISResourceAdapterImpl();
        BISManagedConnectionFactoryImpl managedConnectionFactory = new BISManagedConnectionFactoryImpl();
            
        try {
           resourceAdapter.start(null);
        } catch ( ResourceAdapterInternalException e ) {
           throw new BISException( e.getMessage(), e );
        }
        
        /*
         * Next, the connection properties passed in the connectionProperties parameter will be 
         * processed.  Some of the properties will be used to initialize the BISResourceAdapterImpl 
         * instance and others will be used to initialize the BISManagedConnectionFactoryImpl 
         * instance.  Loop through all of the connction properties and initialize the appropriate 
         * object based on the connection property ID.
         */
        Set keys = connectionProperties.keySet();
        Iterator it = keys.iterator();
            
        Integer key;
        Object value;

        while ( it.hasNext()) {
           key = (Integer)it.next();
           value = connectionProperties.get(key);
                
           switch ( key.intValue() ) {
                
              case BIS_BLOCK_SIZE:
                 try {
                    managedConnectionFactory.setBlockSize( (Integer)value );
                 } catch ( InvalidPropertyException e ) {
                    throw new BISException(e.getMessage(), e);
                 }
                 break;
              case BIS_CHARACTER_SET:
                 managedConnectionFactory.setCharacterSet( (String)value );
                 break;
              case BIS_CONNECTION_TYPE:
                 try {
                    resourceAdapter.setConnectionType( (String)value );
                 } catch ( ResourceException e ) {
                    throw new BISException(e.getMessage(), e);
                 }
                 break;
              case BIS_DEPARTMENT:
                  managedConnectionFactory.setDepartment( (String)value );
                  break;
              case BIS_LOCALE:
                  try {
                      resourceAdapter.setLocale( (String)value );
                  } catch ( ResourceException e ) {
                      throw new BISException(e.getMessage(), e);
                  }
                  break;
              case BIS_PASSWORD:
                  managedConnectionFactory.setPassword( (String)value );
              break;
              case BIS_PORT_NUMBER:
                  resourceAdapter.setPortNumber( (String)value );
                  break;
              case BIS_SERVER_NAME:
                  resourceAdapter.setServerName( (String)value );
                  break;
              case BIS_SITE:
                  managedConnectionFactory.setSite( (String)value );
                  break;
              case BIS_USER_NAME:
                  managedConnectionFactory.setUserName( (String)value );
                  break;
            }
        }
        
        /*
         * Next, associated the BISResourceAdapterImpl instance with the 
         * BISManagedConnectionFactoryImpl instance.
         */
        try {
            managedConnectionFactory.setResourceAdapter( resourceAdapter );
        } catch ( ResourceException e ) {
            throw new BISException(e.getMessage(), e);
        }

        /*
         * Finally, create a BISConnectionFactory instance by calling the createConnectionFactory 
         * method and return the BISConnectionFactory instance.
         */
        
        BISConnectionFactory connectionFactory = null;
        
        try {
            connectionFactory = (BISConnectionFactory)managedConnectionFactory.createConnectionFactory();
        } catch ( ResourceException e ) {
            throw new BISException(e.getMessage(), e);
        }
        
        return connectionFactory;
    }
}
