/*
 * Module:       com.unisys.bis.BISDataset.java
 *
 * Description:  BISDataset interface
 *
 * Product:      Resource Adapter for BIS
 *
 * Date:         February, 2005
 *
 * Copyright (c) 2005 Unisys Corporation.
 *
 * All rights reserved.
 *
 * UNISYS CONFIDENTIAL
 */

package com.unisys.bis;

/**
 * The BISDataset interface defines the methods used to access the data in a BIS dataset.  
 * 
 * A dataset maintains a pointer to the current record in the dataset.  The dataset must 
 * be positioned to a record before data in the record can be accessed.  When a dataset 
 * is initially opened, the current record position is not defined, so the firstRecord() 
 * method or nextRecord() method must be called to initialize the current record position.
 * 
 * @author lamberna
 * 
 */
public interface BISDataset {
    
    /**
     * Called to close a dataset.  This will cause all locks and other resource associated 
     * with the dataset to be released.
     * 
     * @throws BISException - thrown if a dataset can not be closed
     */
    public void close() throws BISException;
    
    /**
     * Called to make the first record in the dataset the current record.
     * 
     * @return false if the datset is empty; otherwise true
     */
    public boolean firstRecord() throws BISException;
    
    /**
     * Called to retrieve the record number of the current record.
     * 
     * @return the current record number; -1 if the current record position has not been set
     */
    public int getCurrentRecordPosition() throws BISException;
    
    /**
     * Called to retrieve the metadata that describes the dataset.  This consists of the 
     * number of columns, and the name and size of each column.
     * 
     * @return a BISDatasetMetaData instance
     * 
     * @throws BISException - thrown if dataset metadata can not be read.
     */
    public BISDatasetMetaData getMetaData() throws BISException;
    
    /**
     * Called to get the value of the designated column in the current record.
     * 
     * @param columnIndex integer value that identifies a column.  The first column is 1.
     * 
     * @return the column value as a String instance
     * 
     * @throws BISException - BISException  thrown if the columnIndex is not valid.
     */
    public String getString(int columnIndex) throws BISException;
    
    /**
     * Called to get the value of the designated column in the current record.
     * 
     * @param columnName a String value that identifies a column.
     * 
     * @return the column value as a String instance.
     * 
     * @throws BISException - thrown if the columnName is not valid.
     */
    public String getString(String columnName) throws BISException;
    
    /**
     * Called to make the last record in the dataset the current record.
     * 
     * @return false if the datset is empty; otherwise true.
     * 
     * @throws BISException  thrown if a dataset read error occurs.
     */
    public boolean lastRecord() throws BISException;
    
    /**
     * Called to make the next record in the dataset the current record.
     * 
     * @return true if the next record exists; otherwise false.
     * 
     * @throws BISException  thrown if a dataset read error occurs.
     */
    public boolean nextRecord() throws BISException;
    
    /**
     * Called to make the previous record in the dataset the current record.
     * 
     * @return true if the previous record exists; otherwise false.
     */
    public boolean previousRecord() throws BISException;
    
    /**
     * Called to set the current record position.  The specified recordPosition must be a 
     * valid record number in the dataset, or an exception will be thrown.
     * 
     * @param recordPosition an integer value that identifies a record.
     * 
     * @throws BISException  thrown if the recordPosition is not valid.
     */
    public void setCurrentRecordPosition(int recordPosition) throws BISException;
}
