/*
 * Module:       com.unisys.bis.BISConnection.java
 *
 * Description:  BISConnection interface
 *
 * Product:      Resource Adapter for BIS
 *
 * Date:         December, 2004
 *
 * Copyright (c) 2005 Unisys Corporation.
 *
 * All rights reserved.
 *
 * UNISYS CONFIDENTIAL
 */

package com.unisys.bis;

/**
 * The BISConnection interface defines the methods used to access BIS data and scripts.  The 
 * BISConnection is an application-level handle used to isolate an application component from 
 * the physical connection to the BIS system
 * 
 * @author lamberna
 *
 */
public interface BISConnection {
    /**
     * Called to close the application-level handle used to access a BIS system.
     * 
     * @throws BISException
     * 
     */
    public void close() throws BISException;

    /**
     * Called to get a BISBlob instance for the specified name.  This method is 
     * used to access data stored in a binary format.
     * 
     * @param name the name of a BISBlob.
     * 
     * @return a BISBlob instance. 
     * 
     * @throws BISException - thrown if an error occurs while opening a BLOB.
     */
    public BISBlob getBLOB(String name) throws BISException;
    
    /**
     * Called to get a BISClob instance for the specified name.  This method is used 
     * to access character data in a dataset as a stream of bytes.
     * 
     * @param name the name of ae BISClob.
     * 
     * @return a BISClob instance.
     * 
     * @throws BISException BISException  thrown if an error occurs while opening a CLOB.
     */
    public BISClob getCLOB(String name) throws BISException;

    /**
     * Called to get a BISDataset instance for the specified dataset name.  
     * The dataset can have either a tabular data format or a free data format.  
     * This method should not be used to open datasets stored using a binary format.
     * 
     * @param datasetName - the name of the dataset
     * 
     * @return a BISDataset instance 
     * 
     * @throws BISException - thrown if an error occurs while opening a dataset.
     */
    public BISDataset getDataset(String datasetName) throws BISException; 
    
    /**
     * Called to get a BISScript instance for the specified script name.  
     * A BISScript instance is used to execute a BIS script.
     * 
     * @param scriptName a script name.
     * 
     * @return a BISScript instance.
     * 
     * @throws BISException thrown if an error occurs while executing a BIS script.
     */
    public BISScript getScript(String scriptName) throws BISException;
    
    /**
     * Called to get a BISScript instance associated with the specified script name 
     * and function name.  A BISScript instance is use to execute a Java script.
     * 
     * @param scriptName a script name.
     * @param functionName a function name.
     * 
     * @return a BISScript instance.
     * 
     * @throws BISException thrown if an error occurs while executing a BIS script.
     */
    public BISScript getScript( String scriptName, String functionName)
                         throws BISException;
}
