/*
 * Module:       com.unisys.os2200.dms.DMSXAConnection.java
 *
 * Description:  Interface which defines the methods used to establish a connection to a 
 *               DMS database.  The DMSXAConnection is an application-level handle used to isolate an 
 *               application component from the physical connection to the DMS system.  This interface 
 *               should be used when participating in a global transaction.
 *               
 * Product:      Resource Adapter for DMS
 *
 * Date:         May, 2007
 *
 * Copyright (c) 2009 Unisys Corporation.
 *
 * All rights reserved.
 *
 * UNISYS CONFIDENTIAL
 */

package com.unisys.os2200.dms;

import java.util.Map;

/**
 * The DMSXAConnection interface defines the methods used to establish a connection to a DMS database.  
 * The DMSXAConnection is an application-level handle used to isolate an application component from the 
 * physical connection to the DMS system.  This interface should be used when participating in a global 
 * transaction.
 */
public interface DMSXAConnection {
    
    /**
     * Indicates that an INVOKE key is needed to access a subschema.
     * Value is the string value defined in the subschema for the INVOKE key.
     * <p>
     * Value Object: java.lang.String
     */
    static final int DMS_INVOKE_KEY = 1;

    /**
     * Indicates that an IMPART key is needed to connect to a subschema.
     * Value is the string value defined in the schema for the IMPART key.
     * <p>
     * Value Object: java.lang.String
     */
    static final int DMS_IMPART_KEY = 2;
    
    /**
     * Indicates if exclusive page locking is enabled when getting records.
     */ 
    static final int DMS_FETCH_WITH_LOCK = 3;

    /**
     * Indicates if queuing is enabled when a page or area is not
     * immediately available.
     */ 
    static final int DMS_QUEUING = 4;

    /*  Fetch With Lock Settings */

    /**
     * Disable exclusive page lock when getting a record.
     */
    static final int NO_FETCH_WITH_LOCK = 0;
    
    /**
     * Enable exclusive page lock when getting a record.  
     * This setting remains in effect until the DMSDatabaseManager 
     * instance is closed.
     */
    static final int STATIC_FETCH_WITH_LOCK = 1;
    
    /**
     * Enable exclusive page lock when getting a record.
     * This setting can be changed during the life of a 
     * DMSDatabaseManager instance.
     */
    static final int DYNAMIC_FETCH_WITH_LOCK = 2;

    /*  No-Queue Settings */

    /**
     * Enable request queuing when a page or area is not immediately
     * available.
     */
    static final int QUEUING_ENABLED = 0;
    
    /**
     * Return an error when a page or area is not immediately
     * available.  
     */    
    static final int QUEUING_DISABLED = 1;

    /**
     * Returns a DMSDatabaseManager instance associated with the specified database name.
     * 
     * @param databaseName name of the DMS database to access 
     * @return a DMSDatabaseManager instance
     * @throws DMSException - thrown if an error occurs while getting a DMSDatabaseManager instance
     * @throws IllegalArgumentException  thrown if the databaseName argument is null
     */
    public DMSDatabaseManager getDatabaseManager(String databaseName)
                                          throws DMSException;

    /**
     * Returns a DMSDatabaseManager instance associated with the specified database name.
     * 
     * @param databaseName name of the DMS database to access 
     * @param configuration a Map instance that contains configuration information needed to
     *                      acquire a DMSDatabaseManager instance.
     * @return a DMSDatabaseManager instance
     * @throws DMSException - thrown if an error occurs while getting a DMSDatabaseManager
     * @throws IllegalArgumentException  thrown if the databaseName argument is null
     */
    public DMSDatabaseManager getDatabaseManager(String databaseName,
                                                 Map<Integer, Object> configuration)
                                          throws DMSException;

    /**
     * Called to close the application-level handle used to access a DMS system.
     * 
     * @throws DMSException - thrown if an error occurs while closing the connection
     */
    public void close() throws DMSException;

}
