/*
* Module:       com.unisys.os2200.dms.DMSResourceAdapter.java
*
* Description:  The DMSResourceAdapter class is used in an unmanaged environment
*               to create a DMSConnectionFactory instance.
*               
* Product:      Resource Adapter for DMS
*
* Date:         April, 2006
*
* Copyright (c) 2010 Unisys Corporation.
*
* All rights reserved.
*
* UNISYS CONFIDENTIAL
*/

package com.unisys.os2200.dms;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.Map.Entry;

import javax.resource.ResourceException;

import com.unisys.os2200.dms.impl.DMSResourceAdapterImpl;
import com.unisys.os2200.dms.impl.DMSManagedConnectionFactoryImpl;

/**
 * The DMSResourceAdapter class is used in an unmanaged environment to create a
 * DMSConnectionFactory instance.
 */
public class DMSResourceAdapter {
    /*
     * This value must match the DMSMessageKey.NULL_PARAMATER_VALUE assignment value
     */
    static final String NULL_PARAMETER_VALUE = "0003";
    
    /**
     * The type of access allowed in the database.
     * Possible values are UPDATE or READ.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: UPDATE
     */
    public static final int DMS_ACCESS_TYPE = 1;
    
    /**
     * The name of the file where updates are written when in TEST or TEST-TRAINING mode.
     * <p>
     * Value Object: java.lang.String
     */
   public static final int DMS_CHANGE_FILE = 2;
   
    /**
     * The character set name used for processing character data in a DMS record
     * item declared as USAGE DISPLAY.  In addition, this character set name is
     * used for processing error message text.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: ISO8859-1
     */
    public static final int DMS_CHARACTER_SET = 3;
    
    /**
     * A string that identifies the connection type. Possible values are 
     * SOCKETS and SECURE-SOCKETS.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: SOCKETS
     */
    public static final int DMS_CONNECTION_TYPE = 4;
    
    /**
     * Indicates if exclusive page locking is enabled when getting records.
     * Possible values are DISABLED, STATIC and DYNAMIC.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: DISABLED
     */
    public static final int DMS_FETCH_WITH_LOCK = 5;
    
    /**
     * Indicates an exception should be thrown when storing a record that 
     * contains a numeric item with a non-numeric value.  The alternative
     * is to provide a valid default value for the numeric item.
     * Possible values are DISABLED and ENABLED.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: ENABLED
     */
    public static final int DMS_ITEM_VALUE_ERROR = 6;
    
    /**
     * Java locale designator.  The locale is typically a 2-character or
     * 5-character string containing a 2-character lowercase language code,
     * optionally followed by an undercore and a 2-character uppercase
     * country code.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * For example: en_US, de, de_CH.
     */
    public static final int DMS_LOCALE = 7;
    
    /**
     * The mode of the program when accessing the database.  Possible values are
     * PRODUCTION, TEST, TRAINING and TEST-TRAINING.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: PRODUCTION
     */
    public static final int DMS_MODE = 8;
    
    /**
     * The password associated with the user name.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Maximum length: 18
     */
    public static final int DMS_PASSWORD = 9;
    
    /**
     * The port number used to connect to the JDMS Server.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: xxxx
     */
    public static final int DMS_PORT_NUMBER = 10;
    
    /**
     * Indicate if queuing is enabled when a page or area is not
     * immediately available.
     * Possible values are ENABLED and DISABLED.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: ENABLED
     */
    public static final int DMS_QUEUING = 11;
    
    /**
     * The type of recovery performed when updating the database.
     * Possible values are QUICKLOOKS and DEFERRED.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: DEFERRED
     */
    public static final int DMS_RECOVERY_OPTION = 12;
    
    /**
     * The name or IP address of the 2200 system on 
     * which the JDMS Server is running.  A list
     * of names, or IP addresses can be specified
     * to enable connection failover.  The valid
     * delimiters are comma, and semicolon.
     * <p>
     * Value Object: java.lang.String
     */
    public static final int DMS_SERVER_NAME = 13;
    
    /**
     * Indicate if UDS thread reuse is permitted.
     * Possible values are DISABLED and ENABLED.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: DISABLED
     */
    public static final int DMS_THREAD_REUSE = 14;
    
    /**
     * The user-id for accessing DMS. 
     * <p>
     * Value Object: java.lang.String
     */
    public static final int DMS_USER_NAME = 15;
    
    /**
     * Indicates if subschema timestamp checking will be performed.
     * Possible values are DISABLED and ENABLED.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: DISABLED
     */
    public static final int DMS_SUBSCHEMA_TIMESTAMP_CHECKING = 16;
    
    /**
     * The fill character used to pad String item values when performing
     * database operations.
     * <p>
     * Possible values are SPACE, LOW_VALUE, and HIGH_VALUE.
     * <p>
     * Value Object: java.lang.String
     * <p>
     * Default value: SPACE
     */
    public static final int DMS_FILL_CHARACTER = 17;
    
    /**
     * The character set name used for processing character data in a DMS record
     * item declared as USAGE DISPLAY-2.
     * <p>
     * Value Object: java.lang.String
     */
    public static final int DMS_CHARACTER_SET_D2 = 18;
    
    /**
     * A static method called to create a DMSConnectionFactory instance when an application is 
     * executed in an unmanaged environment.
     * <p>  
     * The connectionProperties parameter is a Map instance that contains key-value pairs for the
     * connection properties, where the keys are the static final constants defined in this class.
     * The connection properties are used to initialize the DMSConnectionFactory instance that is returned.
     * <p>
     * The implementation will construct a DMSResourceAdapterImpl instance and a DMSManagedConnectionFactoryImpl 
     * instance.  The DMSResourceAdapterImpl.start() method is called to execute startup logic. Also, associate 
     * the DMSResourceAdapterImpl instance with the DMSManagedConnectionFactoryImpl instance.  This needs to be 
     * done before setting any configuration parameters so that error message handling is available if an invalid 
     * configuration parameter value is specified.
     * @param connectionProperties a Map instance that contains key-value pairs for the connection
     *    properties
     * @return a DMSConnectionFactory instance
     * @throws DMSException thrown if an error occurs while creating a DMSConnectionFactory instance
     * @throws IllegalArgumentException thrown if the connectionProperties argument is null
     */
    public static DMSConnectionFactory createConnectionFactory(Map<Integer, String> connectionProperties)
                                                        throws DMSException {
        
        // Construct a DMSResourceAdapterImpl instance and a DMSManagedConnectionFactoryImpl instance.
        // The DMSResourceAdapterImpl.start() method is called to execute startup logic.
        DMSResourceAdapterImpl resourceAdapter = new DMSResourceAdapterImpl();
        DMSManagedConnectionFactoryImpl managedConnectionFactory = new DMSManagedConnectionFactoryImpl();
        
        try {
            resourceAdapter.start(null);
            
            // Associate the DMSResourceAdapterImpl instance with the
            // DMSManagedConnectionFactoryImpl instance
            managedConnectionFactory.setResourceAdapter(resourceAdapter);
        }
        catch (ResourceException e) {
            throw new DMSException(e.getMessage(), e);   
        }
        
        if ( connectionProperties == null ) {
            throw new IllegalArgumentException(resourceAdapter.retrieveText(NULL_PARAMETER_VALUE));          
        }
        
        // Process the connection properties contained in the connectionProperties parameter.
        // Some of the properties will be used to initialize the DMSResourceAdapterImpl instance
        // and others will be used to initialize the DMSManagedConnectionFactoryImpl instance.
        // A loop is used to iterate through all of the connection properties and initialize
        // the appropriate object based on the connection property ID.
        Set<Entry<Integer,String>> entries = connectionProperties.entrySet();
        Iterator<Entry<Integer, String>> it = entries.iterator();
        
        Integer key;
        Object value;
        Map.Entry<Integer, String> map_entry;
        
        while ( it.hasNext() ) {
            map_entry = (Entry<Integer, String>) it.next();
            key = (Integer)map_entry.getKey();
            value = map_entry.getValue();
            
            switch ( key.intValue() ) {
                case DMS_ACCESS_TYPE:
                    try {
                        managedConnectionFactory.setAccessType((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);
                    }
                    break;
                case DMS_CHANGE_FILE:
                    managedConnectionFactory.setChangeFile((String)value);
                    break;
                case DMS_CHARACTER_SET:
                    try {
                        managedConnectionFactory.setCharacterSet((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);   
                    }
                    break;
                case DMS_CONNECTION_TYPE:
                    try {
                        managedConnectionFactory.setConnectionType((String)value);
                    }
                    catch (ResourceException e ) {
                        throw new DMSException(e.getMessage(), e);   
                    }
                    break;
                case DMS_FETCH_WITH_LOCK:
                    try {
                        managedConnectionFactory.setFetchWithLock((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);
                    }
                    break;
                case DMS_FILL_CHARACTER:
                    try {
                        managedConnectionFactory.setFillCharacter((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);
                    }
                    break;
                case DMS_ITEM_VALUE_ERROR:
                    try {
                        managedConnectionFactory.setItemValueError((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);
                    }
                    break;
                case DMS_LOCALE:
                    try {
                        resourceAdapter.setLocale((String)value);
                    }
                    catch (ResourceException e ) {
                        throw new DMSException(e.getMessage(), e);   
                    }
                    break;
                case DMS_MODE:
                    try {
                        managedConnectionFactory.setMode((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);
                    }
                    break;
                case DMS_PASSWORD:
                    managedConnectionFactory.setPassword((String)value);
                    break;
                case DMS_PORT_NUMBER:
                    managedConnectionFactory.setPortNumber((String)value);
                    break;
                case DMS_QUEUING:
                    try {
                        managedConnectionFactory.setQueuing((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);
                    }
                    break;
                case DMS_RECOVERY_OPTION:
                    try {
                        managedConnectionFactory.setRecoveryOption((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);
                    }
                    break;
                case DMS_SERVER_NAME:
                    managedConnectionFactory.setServerName((String)value);
                    break;
                case DMS_SUBSCHEMA_TIMESTAMP_CHECKING:
                    try {
                        managedConnectionFactory.setSubschemaTimestampChecking((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);
                    }
                    break;
                case DMS_THREAD_REUSE:
                    try {
                        managedConnectionFactory.setThreadReuse((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);
                    }
                    break;
                case DMS_USER_NAME:
                    managedConnectionFactory.setUserName((String)value);
                    break;
                case DMS_CHARACTER_SET_D2:
                    try {
                        managedConnectionFactory.setCharacterSetD2((String)value);
                    } catch (ResourceException e) {
                        throw new DMSException(e.getMessage(), e);   
                    }
                    break;
             }
        }
        
        // Create a DMSConnectionFactory instance by calling the createConnectionFactory method
        // and then return the DMSConnectionFactory instance.
        DMSConnectionFactory connectionFactory = null;
        
        try {
            connectionFactory = (DMSConnectionFactory)managedConnectionFactory.createConnectionFactory();   
        }
        catch (ResourceException e) {
            throw new DMSException(e.getMessage(), e);   
        }
        
        return connectionFactory;
    }
    
}
