/*
* Module:       com.unisys.os2200.dms.DMSRecordSet.java
*
* Description:  The DMSRecordSet interface is used to represent the relationship that exists
*               between an owner record type and one or more member record types.
*               
* Product:      Resource Adapter for DMS
*
* Date:         April, 2006
*
* Copyright (c) 2009 Unisys Corporation.
*
* All rights reserved.
*
* UNISYS CONFIDENTIAL
*/

package com.unisys.os2200.dms;

import java.util.List;
import java.util.Map;

/**
 * The DMSRecordSet interface is used to represent the relationship that exists between
 * an owner record type and one or more member record types.
 */
public interface DMSRecordSet {

    /**
     * Finds the first record in the set with the specified item keys and makes it
     * the selected record.
     * @param recordType the type of record
     * @param searchKeys a java.util.Map which contains key/value pairs for the item keys to search for
     * @return true if the record exists; otherwise returns false
     * @throws DMSException thrown if an error occurs while selecting a record
     * @throws IllegalArgumentException thrown if the searchKeys argument is null
     */
    public boolean findRecord(int recordType,
                              Map<Integer, Object> searchKeys) 
                       throws DMSException;
	
    /**
     * Called to find the first record in the set with the specified item keys 
     * and make it the selected record.
     * @param recordType type of record
     * @param searchKeys key/value pairs used to search for a record
     * @param charsetNames character set names
     * @return return true if the record exists; otherwise return false
     * @throws DMSException Thrown if an error occurs while selecting a record
     * @throws IllegalArgumentException thrown if the searchKeys argument is null
     */
    public boolean findRecord(int recordType,
                              Map<Integer, Object> searchKeys,
                              Map<Integer, String> charsetNames)
                       throws DMSException;
    
    /**
     * Makes the first record in the set the selected record.
     * @return true if the record exists; otherwise return false
     * @throws DMSException thrown if an error occurs while selecting a record
     */
    public boolean firstRecord()
                        throws DMSException;
	
    /**
     * Makes the first record of type recordType in the set the selected record. 
     * @param recordType the type of record
     * @return true if the record exists; otherwise returns false
     * @throws DMSException thrown if an error occurs while selecting a record
     */
    public boolean firstRecord(int recordType)
                        throws DMSException;
	
    /**
     * Called to get a list of database keys.
     * <p>
     * Note: Only pointer array and indexed pointer array recordSets support this method.
     * @param numberOfKeys indicates the number of database keys to retrieve
     * @param lastRetrievedKey a DMSDatabaseKey instance or null 
     * @return a DMSDatabaseKeyList instance
     * @throws DMSException  thrown if an error occurs while getting a database key list.
     */
    public DMSDatabaseKeyList getDatabaseKeyList(int numberOfKeys,
                                                 DMSDatabaseKey lastRetrievedKey)
                                          throws DMSException;
    
    /**
     * Called to get a list of database keys.
     * <p>
     * Note: Only pointer array and indexed pointer array recordSets support this method.
     * @param numberOfKeys indicates the number of database keys to retrieve
     * @param lastRetrievedKey a DMSDatabaseKey instance or null 
     * @return a DMSDatabaseKeyList instance
     * @throws DMSException  thrown if an error occurs while getting a database key list.
     */
    public DMSDatabaseKeyList getDefinedKeyList(int numberOfKeys,
                                                DMSDatabaseKey lastRetrievedKey)
                                         throws DMSException;
    
    /**
     * Gets the owner record of the set.
     * @return a DMSRecord instance
     * @throws DMSRecordNotFoundException  thrown if the requested record can not be found
     * @throws DMSException thrown if an error occurs while retrieving the owner record
     */
    public DMSRecord getOwnerRecord()
                             throws DMSRecordNotFoundException, DMSException;
	
    /**
     * Gets the selected record of the set.
     * @return a DMSRecord instance
     * @throws DMSException thrown if a record in the set has not been selected
     */
    public DMSRecord getSelectedRecord()
                                throws DMSException;
	
    /**
     * Inserts the specified record into the manual set.  The insertion point 
     * is determined by the schema set order.  If the set order is NEXT or PREVIOUS, 
     * then the insertion point is determined by the selected record.
     * @param record a DMSRecord instance
     * @throws DMSException thrown if an error occurs while inserting a record into the set
     * @throws IllegalArgumentException thrown if the record argument is null
     */
    public void insertRecord(DMSRecord record)
                      throws DMSException;
    
    /**
     * Determines if the set contains any records.
     * @return true if the set is empty; otherwise returns false
     * @throws DMSException thrown if an error occurs
     */
    public boolean isEmpty()
                    throws DMSException;
    
    /**
     * Makes the last record in the set the selected record.
     * @return true if the record exists; otherwise returns false
     * @throws DMSException thrown if an error occurs while selecting a record
     */
    public boolean lastRecord()
                       throws DMSException;
    
    /**
     * Makes the last record of type recordType in the set the selected record.
     * @param recordType the type of record
     * @return true if the record exists; otherwise returns false
     * @throws DMSException
     */
    public boolean lastRecord(int recordType)
                       throws DMSException;
    
    /**
     * Locates a record in the set that has record item values that equal the record item values
     * of the currently selected record.  The record item identifiers in the specified list of keys
     * determines which record items' values to compare.  If a duplicate record is found, it
     * it is now the selected record.
     * @param searchKeys a java.util.List which contains record item identifiers that will be used as keys
     * @return true if the record exists; otherwise returns false
     * @throws DMSException thrown if an error occurs while locating a record
     * @throws IllegalArgumentException thrown if the searchKeys argument is null
     */
    public boolean nextDuplicateRecord(List<Integer> searchKeys)
                                throws DMSException;
    
    /**
     * Makes the next record in the set the selected record.
     * @return true if the record exists; otherwise returns false
     * @throws DMSException thrown if an error occurs while selecting a record
     */
    public boolean nextRecord()
                       throws DMSException;
    
    /**
     * Makes the next record of type recordType in the set the selected record.
     * @param recordType the type of record
     * @return true if the record exists; otherwise returns false
     * @throws DMSException thrown if an error occurs while selecting a record
     */
    public boolean nextRecord(int recordType)
                       throws DMSException;
    
    /**
     * Makes the previous record in the set the selected record.
     * @return true if the record exists; otherwise returns false
     * @throws DMSException thrown if an error occurs while selecting a record
     */
    public boolean previousRecord()
                           throws DMSException;
    
    /**
     * Makes the previous record of type recordType in the set the selected record.
     * @param recordType the type of record
     * @return true if the record exists; otherwise returns false
     * @throws DMSException thrown if an error occurs while selecting a record
     */
    public boolean previousRecord(int recordType)
                           throws DMSException;
    
    /**
     * Removes the selected record from the manual set.
     * @throws DMSException thrown if an error occurs while removing the record
     */
    public void removeRecord()
                      throws DMSException;
    
    /**
     * Removes the selected record from the manual set.  If the type of the selected 
     * record does not match the specified recordType, then an exception is thrown.
     * @param recordType used to designate the type of the record to remove
     * @throws DMSException thrown if an error occurs while removing the record
     */
    public void removeRecord(int recordType) 
                      throws DMSException;
    
}
