/*
* Module:       com.unisys.os2200.dms.DMSRecord.java
*
* Description:  The DMSRecord interface is used to represent a record in the DMS database.
*               
* Product:      Resource Adapter for DMS
*
* Date:         April, 2006
*
* Copyright (c) 2009 Unisys Corporation.
*
* All rights reserved.
*
* UNISYS CONFIDENTIAL
*/

package com.unisys.os2200.dms;

import java.util.List;
import java.math.BigDecimal;
import java.math.BigInteger;

/**
 * The DMSRecord interface is used to represent a record in the DMS database.
 */
public interface DMSRecord {
    
    /**
     * Append a new repeating record item.
     * @param parentHandle a unique value that identifies the parent of a record item.
     * @param itemCode the subschema code of the repeating record item to append.
     * @param odoItemIndex a unique value that identifies the DEPENDING ON item.
     * @return index of the new repeating record item
     * @throws DMSException thrown if an error occurs while appending a repeating record item
     */
    public int addItem(int parentHandle,
                       int itemCode,
                       int odoItemIndex)
                throws DMSException;
    
    /**
     * Insert a new repeating record item at the specified location.
     * @param parentHandle a unique value that identifies the parent of a record item.
     * @param itemCode the subschema code of the repeating record item to insert.
     * @param itemIndex position where the new repeating record item is inserted.
     * @param odoItemIndex a unique value that identifies the DEPENDING ON item.
     * @return index of the new repeating record item
     * @throws DMSException thrown if an error occurs while inserting a repeating record item
     */
    public int addItem(int parentHandle,
                       int itemCode,
                       int itemIndex,
                       int odoItemIndex)
                throws DMSException;

    /**
     * Deletes the record.  If the record is the owner of a set, the automatic members
     * of the set are also deleted.
     * @throws DMSException thrown if an error occurs while deleting the record
     */
    public void delete()
                throws DMSException;
	
    /**
     * Deletes the record.  If the record is the owner of a set, then automatic and
     * manual members of the set are also deleted.
     * @throws DMSException thrown if an error occurs while deleting the record
     */	
    public void deleteAll()
                   throws DMSException;
	
    /**
     * Deletes the record.  If the record is the owner of a set, the set must not
     * contain any automatic or manual members.
     * @throws DMSException thrown if an error occurs while deleting the record
     */
    public void deleteIfEmpty()
                       throws DMSException;
    
    /**
     * Retrieves the value of the designated record item as a BigDecimal value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public BigDecimal getBigDecimal(int itemIndex)
                             throws DMSItemValueException, DMSException;
    
    /**
     * Retrieves the value of the designated record item as a BigDecimal value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public BigDecimal getBigDecimal(int parentHandle,
                                    int itemIndex)
                             throws DMSItemValueException, DMSException;
    
    /**
     * Retrieves the value of the designated record item as a BigInteger value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public BigInteger getBigInteger(int itemIndex)
                             throws DMSException; 

    /**
     * Retrieves the value of the designated record item as a BigInteger value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public BigInteger getBigInteger(int parentHandle,
                                    int itemIndex)
                             throws DMSException;

    /**
     * Retrieves the database key associated with the record.
     * @return a DMSDatabaseKey instance
     * @throws DMSException thrown if an error occurs while retrieving the database key
     */
    public DMSDatabaseKey getDatabaseKey()
                                  throws DMSException;
	
    /**
     * Retrieves the value of the designated record item as a double value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public double getDouble(int itemIndex)
                     throws DMSException;

    /**
     * Retrieves the value of the designated record item as a double value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public double getDouble(int parentHandle,
                            int itemIndex)
                     throws DMSException;

    /**
     * Retrieves the value of the designated record item as a float value.
     * @param itemIndex The record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public float getFloat(int itemIndex)
                   throws DMSException;
    
    /**
     * Retrieves the value of the designated record item as a float value.
     * @param parentHandle a unique value that identifies the parent of a record item.
     * @param itemIndex The record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public float getFloat(int parentHandle,
                          int itemIndex)
                   throws DMSException;
    
    /**
     * Retrieves the value of the designated record item as an int value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public int getInt(int itemIndex)
               throws DMSItemValueException, DMSException;
	
    /**
     * Retrieves the value of the designated record item as an int value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public int getInt(int parentHandle,
                      int itemIndex)
               throws DMSItemValueException, DMSException;
    
    /**
     * Retrieves the value of the designated record item as a long value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSItemValueException  thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public long getLong(int itemIndex)
                 throws DMSItemValueException, DMSException;
    
    /**
     * Retrieves the value of the designated record item as a long value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public long getLong(int parentHandle,
                        int itemIndex)
                 throws DMSItemValueException, DMSException;
	
    /**
     * Retrieves a DMSRecordSet instance for the specified recordSetID.
     * @param recordSetID the subschema identifier of a recordSet
     * @return a DMSRecordSet instance
     * @throws DMSException thrown if an error occurs while creating the DMSRecordSet instance
     */
    public DMSRecordSet getRecordSet(int recordSetID)
                              throws DMSException;

    /**
     * Retrieves the value of the designated record item as a short value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public short getShort(int itemIndex)
                   throws DMSItemValueException, DMSException;
    
    /**
     * Retrieves the value of the designated record item as a short value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public short getShort(int parentHandle,
                          int itemIndex)
                   throws DMSItemValueException, DMSException;
	
    /**
     * Retrieves the value of the designated record item as a java.lang.String value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public String getString(int itemIndex)
                     throws DMSException;
    
    /**
     * Retrieves the value of the designated record item as a java.lang.String value.
     * @param itemIndex the record item number.  The first record item is 1
     * @param characterSet a character set name
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public String getString(int itemIndex,
                            String characterSet)
                     throws DMSException;
	
    /**
     * Retrieves the value of the designated record item as a java.lang.String value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public String getString(int parentHandle,
                            int itemIndex)
                     throws DMSException;
    
    /**
     * Retrieves the value of the designated record item as a java.lang.String value.
     * @param parentHandle a unique value that identifies the parent of a record item.
     * @param itemIndex the record item number.  The first record item is 1.
     * @param characterSet a character set name.
     * @return the item value
     * @throws DMSException thrown if an error occurs while accessing the record item
     */
    public String getString(int parentHandle,
                            int itemIndex,
                            String characterSet)
                     throws DMSException;
    
    /**
     * Called to determine if any record items contain changes that have not been saved in 
     * the database.
     * @return return true if the DMSRecord object contains changes; otherwise, return false
     */
    public boolean hasPendingChanges();
        
    /**
     * Called to determine if this DMSrecord represents a newly created record that has not 
     * been stored in the database.
     * @return return true if the newly created DMSRecord object has not been stored in the 
     *         database; otherwise, return false
     */
    public boolean isNewRecord();
    
    /**
     * Called to keep the lock on the most recently retrieved DMSRecord to ensure that the 
     * record can not be modified by another application.
     * @throws DMSException thrown if an error occurs while performing a keepLock operation
     */
    public void keepLock()
                  throws DMSException;
    
    /**
     * Updates the record in the database.
     * @throws DMSException thrown if an error occurs while modifying the record
     */
    public void modify()
                throws DMSException;
	
    /**
     * Updates the record in the database.  Also, the record is linked to a new owner
     * in the specified recordSet.  Insertion of the record into the specified recordSet
     * is based on the ORDER clause in the schema and the currently selected record
     * in the recordSet instance.
     * @param recordSet a DMSRecordSet instance
     * @throws DMSException thrown if an error occurs while modifying the record
     * @throws IllegalArgumentException thrown if the recordSet argument is null
     */
    public void modify(DMSRecordSet recordSet)
                throws DMSException;
	
    /**
     * Updates the record in the database.  Also, the record is linked to a new owner
     * in the specified list of recordSets.  Insertion of the record into each recordSet
     * in the list of specified recordSets is based on the ORDER clause in the schema
     * and the currently selected record in each recordSet.
     * @param recordSets a java.util.List instance that contains references to one
     * or more DMSRecordSet instances.
     * @throws DMSException thrown if an error occurs while modifying the record
     * @throws IllegalArgumentException thrown if the recordSets argument is null
     */
    public void modify(List<DMSRecordSet> recordSets)
                throws DMSException;
	
    /**
     * Removes the record from the specified manual recordSet.
     * @param recordSetID a subschema identifier for a record set
     * @throws DMSException thrown if an error occurs while removing the record from 
     * the record set
     */
    public void remove(int recordSetID) 
                throws DMSException;
	
    /**
     * Removes the record from all manual record sets in which the record participates.
     * @throws DMSException
     */
    public void removeAll()
                   throws DMSException;
    
    /**
     * Remove a repeating record item.
     * @param parentHandle unique value that identifies the parent of a record item.
     * @param itemIndex the record item number.  The first record item is 1.
     * @param odoItemIndex a unique value that identifies the DEPENDING ON item.
     * @throws DMSException thrown if an error occurs while removing a repeating record item
     */
    public void removeItem(int parentHandle,
                           int itemIndex,
                           int odoItemIndex)
                    throws DMSException;
    
    /**
     * Remove a range of repeating record items.
     * @param parentHandle a unique value that identifies the parent of a record item.
     * @param fromIndex the record item number of the first item to remove.  The first record item is 1.
     * @param toIndex the record item number after the last item to remove.
     * @param odoItemIndex a unique value that identifies the DEPENDING ON item.
     * @throws DMSException thrown if an error occurs while removing a range of repeating record items
     */
    public void removeItemRange(int parentHandle,
                                int fromIndex,
                                int toIndex,
                                int odoItemIndex)
                         throws DMSException;
    
    /**
     * Sets the value of the designated record item with a BigDecimal value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     * @throws IllegalArgumentException thrown if the value argument is null
     */
    public void setBigDecimal(int itemIndex,
                              BigDecimal value)
                       throws DMSException;
    
    /**
     * Sets the value of the designated record item with a BigDecimal value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     * @throws IllegalArgumentException thrown if the value argument is null
     */
    public void setBigDecimal(int parentHandle,
                              int itemIndex,
                              BigDecimal value)
                       throws DMSException;
	
    /**
     * Sets the value of the designated record item with a BigInteger value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new record item value
     * @throws DMSException thrown if an error occurs while setting the record item
     * @throws IllegalArgumentException thrown if the value argument is null
     */
    public void setBigInteger(int itemIndex,
                              BigInteger value)
                       throws DMSException;

    /**
     * Sets the value of the designated record item with a BigInteger value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new record item value
     * @throws DMSException thrown if an error occurs while setting the record item
     * @throws IllegalArgumentException thrown if the value argument is null
     */
    public void setBigInteger(int parentHandle,
                              int itemIndex,
                              BigInteger value)
                       throws DMSException;

    /**
     * Sets the value of the designated record item with a double value.
     * <p>
     * Valid value range: 2.781342323134002E-309 through 8.988465674311579E+307
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new record item value
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setDouble(int itemIndex,
                          double value)
                   throws DMSException;

    /**
     * Sets the value of the designated record item with a double value.
     * <p>
     * Valid value range: 2.781342323134002E-309 through 8.988465674311579E+307
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new record item value
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setDouble(int parentHandle,
                          int itemIndex,
                          double value)
                   throws DMSException;

    /**
     * Sets the value of the designated record item with a float value.
     * <p>
     * Valid value range: 1.4693687E-39F through 1.70141178E38F
     * @param itemIndex The record item number.  The first record item is 1.
     * @param value the new record item value
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setFloat(int itemIndex,
                         float value)
                  throws DMSException;
    
    /**
     * Sets the value of the designated record item with a float value.
     * <p>
     * Valid value range: 1.4693687E-39F through 1.70141178E38F
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex The record item number.  The first record item is 1.
     * @param value the new record item value
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setFloat(int parentHandle,
                         int itemIndex,
                         float value)
                  throws DMSException;
    
    /**
     * Sets the value of the designated record item with an int value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setInt(int itemIndex,
                       int value)
                throws DMSException;
    
    /**
     * Sets the value of the designated record item with an int value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setInt(int parentHandle,
                       int itemIndex,
                       int value)
                throws DMSException;
	
    /**
     * Sets the value of the designated record item with a long value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setLong(int itemIndex,
                        long value)
                 throws DMSException;
    
    /**
     * Sets the value of the designated record item with a long value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setLong(int parentHandle,
                        int itemIndex,
                        long value)
                 throws DMSException;
	
    /**
     * Sets the value of the designated record item with a short value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setShort(int itemIndex,
                         short value)
                  throws DMSException;
    
    /**
     * Sets the value of the designated record item with a short value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     */
    public void setShort(int parentHandle,
                         int itemIndex,
                         short value)
                  throws DMSException;
	
    /**
     * Sets the value of the designated record item with a java.lang.String value.
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     * @throws IllegalArgumentException thrown if the value argument is null
     */
    public void setString(int itemIndex,
                          String value)
                   throws DMSException;
    
    /**
     * Sets the value of the designated record item with a java.lang.String value.
     * @param itemIndex the record item number.  The first record item is 1
     * @param value the new record item value
     * @param characterSet a character set name
     * @throws DMSException thrown if an error occurs while setting the record item
     * @throws IllegalArgumentException thrown if the value argument is null
     */
    public void setString(int itemIndex,
                          String value,
                          String characterSet)
                   throws DMSException;
    
    /**
     * Sets the value of the designated record item with a java.lang.String value.
     * @param parentHandle a unique value that identifies the parent of a record item.
     * @param itemIndex the record item number.  The first record item is 1
     * @param value the new record item value
     * @param characterSet a character set name
     * @throws DMSException thrown if an error occurs while setting the record item
     * @throws IllegalArgumentException thrown if the value argument is null
     */
    public void setString(int parentHandle,
                          int itemIndex,
                          String value,
                          String characterSet)
                   throws DMSException;
    
    /**
     * Sets the value of the designated record item with a java.lang.String value.
     * @param parentHandle a unique value that identifies the parent of a record item
     * @param itemIndex the record item number.  The first record item is 1.
     * @param value the new value for the record item
     * @throws DMSException thrown if an error occurs while setting the record item
     * @throws IllegalArgumentException thrown if the value argument is null
     */
    public void setString(int parentHandle,
                          int itemIndex,
                          String value)
                   throws DMSException;
	
    /**
     * Stores a new record in the database.
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while storing the record
     */
    public void store()
               throws DMSItemValueException, DMSException;
	
    /**
     * Stores the new record in the database.  This method should only be used if the record
     * is an automatic member of a single set.  Insertion of the new record into the specified
     * recordSet is based on the ORDER clause in the schema and the currently selected record
     * in the recordSet instance.
     * @param recordSet a DMSRecordSet instance
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while storing the record
     * @throws IllegalArgumentException thrown if the recordSet argument is null
     */
    public void store(DMSRecordSet recordSet)
               throws DMSItemValueException, DMSException;
	
    /**
     * Stores the new record in the database.  This method should only be used if the record
     * is an automatic member of multiple record sets.  Insertion of the new record into each
     * recordSet in the list of specified recordSets is based on the ORDER clause in the schema 
     * and the currently selected record in each recordSet.
     * @param recordSets a java.util.List instance that contains references to one or more
     * DMSRecordSet instances
     * @throws DMSItemValueException thrown if a numeric record item contains a non-numeric value
     * @throws DMSException thrown if an error occurs while storing the record
     * @throws IllegalArgumentException thrown if the recordSets argument is null
     */
    public void store(List<DMSRecordSet> recordSets)
               throws DMSItemValueException, DMSException;
	
    /**
     * Transforms DMS record data into an XML document.
     * @return an XML document as a String
     * @throws DMSException thrown if an error occurs while generating an XML document
     */
    public String toXML()
                 throws DMSException;
    
    /**
     * Transforms DMS record data into an XML document, or XML fragment.
     * @param fragment specify true to generate an XML fragment or false to generate an XML document
     * @param encoding XML encoding.  For example, UTF-8
     * @return an XML document, or XML fragment as a String
     * @throws DMSException thrown if an error occurs while generating an XML document
     */
    public String toXML(boolean fragment, String encoding)
                 throws DMSException;

}
