/*
* Module:       com.unisys.os2200.dms.DMSItemValueException.java
*
* Description:  The DMSItemValueException class extends the DMSException class.  This exception 
*               is thrown when the get accessor method for a numeric record item is called and 
*               the record item contains a non-numeric value or when the store method is called 
*               and a record contains numeric items with non-numeric values.
* 
* Product:      Resource Adapter for DMS
*
* Date:         November, 2006
*
* Copyright (c) 2009 Unisys Corporation.
*
* All rights reserved.
*
* UNISYS CONFIDENTIAL
*/

package com.unisys.os2200.dms;

import java.math.BigDecimal;

/**
 * The DMSItemValueException is thrown to indicate the get accessor method for a numeric record 
 * item is called and the record item contains a non-numeric value or when the store method is 
 * called and a record contains numeric items with non-numeric values.
 *
 */
public final class DMSItemValueException extends DMSException {
    
    // Universal version identifier for a Serializable class
    static final long serialVersionUID = 1998937358090701278L;
    
    private Class<?>          itemJavaType;   // A Class instance that represents the type of the numeric 
                                              // record item. For example: short.class
    private static Class<?>[] itemJavaTypes = {null,            // An array of Class instances. This array is used 
                                               null,            // to convert a numeric Java type identifier to a 
                                               short.class,     // Class instance that represents the actual Java 
                                               int.class,       // type.
                                               long.class,
                                               null,            // Note: A value of null in the array indicates the
                                               null,            // Java type cannot contain a bad value or the Java
                                               null,            // type is not used by the DMS resource adapter.
                                               null,
                                               null,
                                               BigDecimal.class,    
                                               null,
                                               null,
                                               null};
    private String            itemValue;      // The record item value represented as a String.

    /**
     * Default class constructor.
     * 
     */
    public DMSItemValueException() {
        super();
    }
    
    /**
     * Class constructor.
     * 
     * @param message  error message text
     * @param itemType  a number that indicates the Java type of the record item
     * @param itemValue  the record item value represented as a String
     */
    public DMSItemValueException(String message,
                                 byte itemType,
                                 String itemValue) {
        super(message);
        
        // Use itemType to get a Class instance from this.itemJavaTypes and save it in this.itemJavaType.
        this.itemJavaType = DMSItemValueException.itemJavaTypes[itemType];
        
        this.itemValue = itemValue;
    }

    /**
     * Class Constructor.
     * 
     * @param cause  a Throwable instance that describes the original cause of the exception
     * @param itemType  a number that indicates the Java type of the record item
     * @param itemValue  the record item value represented as a String
     */
    public DMSItemValueException(Throwable cause, 
                                 byte itemType,
                                 String itemValue) {
        super(cause);
        
        // Use itemType to get a Class instance from this.itemJavaTypes and save it in this.itemJavaType.
        this.itemJavaType = DMSItemValueException.itemJavaTypes[itemType];
        
        this.itemValue = itemValue;
    }

    /**
     * Class constructor.
     * 
     * @param message  error message text
     * @param cause  a Throwable instance that describes the original cause of the exception
     * @param itemType  a number that indicates the Java type of the record item
     * @param itemValue  the record item value represented as a String
     */
    public DMSItemValueException(String message,
                                 Throwable cause,
                                 byte itemType,
                                 String itemValue) {
        super(message, cause);
        
        // Use itemType to get a Class instance from this.itemJavaTypes and save it in this.itemJavaType. 
        this.itemJavaType = DMSItemValueException.itemJavaTypes[itemType];
        
        this.itemValue = itemValue;
    }

    /**
     * Accessor for the itemJavaType property. 
     * 
     * @return a Class instance that represents the java type of the numeric record item 
     */
    public Class<?> getItemJavaType() {
        return this.itemJavaType;
    }
    
    /**
     * Accessor for the itemValue property.
     * 
     * @return the record item value represented as a String
     */
    public String getItemValue() {
        return this.itemValue;
    }
    
    /**
     * Returns a String that contains error information. 
     * 
     * @return a String that contains error information
     */
    public String toString() {
        // The implementation of this class returns a String that contains the properties 
        // of the base class and the this.itemValue property from this class.
        String result = super.toString();

        result = result + " Item Value: " + this.itemValue;
        
        return result;
    }
}
