/*
* Module:       com.unisys.os2200.dms.DMSException.java
*
* Description:  Base class for all exceptions generated by the resource adapter.
*               The main purpose for defining this class is to provide a way for application
*               components to catch exceptions generated by this resource adapter.
*               
* Product:      Resource Adapter for DMS
*
* Date:         April, 2006
*
* Copyright (c) 2006 Unisys Corporation.
*
* All rights reserved.
*
* UNISYS CONFIDENTIAL
*/

package com.unisys.os2200.dms;

/**
 * The DMSException class extends the java.lang.Exception class.  It is the base class for all DMS 
 * related exceptions thrown by the DMS resource adapter. 
 * <p>
 * The main purpose for defining this class is to provide a way for application components to catch 
 * exceptions generated by this resource adapter.
 */
public class DMSException extends Exception {

    // Universal version identifier for a Serializable class
    static final long serialVersionUID = 3037085608204789670L;
    
    private int errorCode; // The error code returned by the DMR
    private int errorCommand; // The identifier of the command that caused an error
    private int errorStatus; // The error status returned by the JDMS server
    private int rollbackErrorCode; // The rollback error code returned by the DMR
    
    /**
     * Class constructor with no arguments.
     */
    public DMSException() {

    	super();
    }

    /**
     * Class constructor with a message parameter.
     * @param message error message text
     */
    public DMSException(String message) {
    	
        super( message );
    }

    /**
     * Class constructor with a Throwable parameter.
     * @param cause a Throwable instance that describes the original cause of the exception
     */
    public DMSException(Throwable cause) {
    	
        super( cause );
    }

    /**
     * Class constructor with message and Throwable parameters.
     * @param message error message text
     * @param cause a Throwable instance that describes the original cause of the exception
     */
    public DMSException(String message, Throwable cause) {
    	
        super( message, cause );
    }

    /**
     * Class constructor with property parameters.
     * @param errorStatus the JDMS server error status
     * @param errorCode the DMR error code
     * @param rollbackErrorCode the DMS rollback error code
     * @param errorCommand the identifier of the command that caused an error
     */
    public DMSException(int errorStatus,
                        int errorCode,
                        int rollbackErrorCode,
                        int errorCommand) {
        super();
        this.errorStatus = errorStatus;
        this.errorCode = errorCode;
        this.rollbackErrorCode = rollbackErrorCode;
        this.errorCommand = errorCommand;
    }

    /**
     * Class constructor with a message parameter and property parameters.
     * @param message error message text
     * @param errorStatus the JDMS server error status
     * @param errorCode the DMR error code
     * @param rollbackErrorCode the DMS rollback error code
     * @param errorCommand the identifier of the command that caused an error
     */
    public DMSException(String message,
                        int errorStatus,
                        int errorCode,
                        int rollbackErrorCode,
                        int errorCommand) {
        super(message);
        this.errorStatus = errorStatus;
        this.errorCode = errorCode;
        this.rollbackErrorCode = rollbackErrorCode;
        this.errorCommand = errorCommand;
    }

    /**
     * Class constructor with a message parameter, a Throwable parameter and property parameters.
     * @param message error message text
     * @param cause a Throwable instance that describes the original cause of the exception
     * @param errorStatus the JDMS server error status
     * @param errorCode the DMR error code
     * @param rollbackErrorCode the DMS rollback error code
     * @param errorCommand the identifier of the command that caused an error
     */
    public DMSException(String message,
                        Throwable cause,
                        int errorStatus,
                        int errorCode,
                        int rollbackErrorCode,
                        int errorCommand) {
        super(message, cause);
        this.errorStatus = errorStatus;
        this.errorCode = errorCode;
        this.rollbackErrorCode = rollbackErrorCode;
        this.errorCommand = errorCommand;
    }

    /**
     * Returns a String that contains error information.
     * <p>
     * The implementation of this class returns a String that contains the 
     * properties of this class.
     * @return a String that contains error information
     */
    public String toString() {
        String result = super.toString();
        if ( errorStatus != 0 ) {
            result = result + " ErrorStatus: " + this.errorStatus;
        } 
        if ( errorCode != 0 ) {
            result = result + " ErrorCode: " + this.errorCode;
        } 
        if ( rollbackErrorCode != 0 ) {
            result = result + " RollbackError: " + this.rollbackErrorCode;
        }
        if ( errorCommand != 0 ) {
            result = result + " ErrorCommand: " + this.errorCommand;
        }
   
        return  result;
    }
    
    /**
     * Get accessor for the JDMS server error status.
     * @return the JDMS server error status value
     */
    public int getErrorStatus() {
        return this.errorStatus;
    }
    
    /**
     * Get accessor for the DMR error code.
     * @return the DMR error code value
     */
    public int getErrorCode() {
        return this.errorCode;
    }
    
    /**
     * Get accessor for the DMS rollback error code.
     * @return the DMS rollback error code value
     */
    public int getRollbackErrorCode() {
        return this.rollbackErrorCode;
    }
    
    /**
     * Get accessor for the identifier of the command that caused an error.
     * @return the identifier of the command that caused an error value
     */
    public int getErrorCommand() {
        return this.errorCommand;
    }
    
}
