/* 
* Module:       com.unisys.os2200.dms.DMSDatabaseManager.java
*
* Description:  Interface which defines the methods used to access a DMS database.
*               
* Product:      Resource Adapter for DMS
*
* Date:         April, 2006
*
* Copyright (c) 2009 Unisys Corporation.
*
* All rights reserved.
*
* UNISYS CONFIDENTIAL
*/

package com.unisys.os2200.dms;

import java.util.Map;
import java.util.Set;

/**
 * The DMSDatabaseManager interface defines the methods used to access a DMS database.
 */
public interface DMSDatabaseManager {
	
    /* Fetch With Lock Settings */

    /**
     * Disables exclusive page lock when getting a record.
     */
    static final int NO_FETCH_WITH_LOCK = 0;
    /**
     * Enables exclusive page lock when getting a record.
     * This setting can be changed during the life of a 
     * DMSDatabaseManager instance.
     */
    static final int DYNAMIC_FETCH_WITH_LOCK = 2;
    
    /* Usage modes for opening an area */
    
    /**
     * Used to open an area for retrieval of records.
     */
    static final int USAGEMODE_RETRIEVAL = 0;
    /**
     * Used to open an area for exclusive retrieval of records.
     */
    static final int USAGEMODE_EXCLUSIVE_RETRIEVAL = 1;
    /**
     * Used to open an area for update of records.
     */
    static final int USAGEMODE_UPDATE = 2;
    /**
     * Used to open an area for exclusive update of records.
     */
    static final int USAGEMODE_EXCLUSIVE_UPDATE = 3;
    /** 
     * Used to open an area for initial load of records.
     */
    static final int USAGEMODE_INITIAL_LOAD = 4;
    /**
     * Used to open an area for protected retrieval of records..
     */
    static final int USAGEMODE_PROTECTED_RETRIEVAL = 8;
    /**
     * Used to open an area for protected update of records.
     */
    static final int USAGEMODE_PROTECTED_UPDATE = 10;
	
    /**
     * Invalidates the DMSDatabaseManager instance.
     * @throws DMSException - thrown if an error occurs while closing access to the database.
     */
    public void close()
               throws DMSException;

    /**
     * Closes all open areas.
     * @throws DMSException - thrown if an error occurs while closing open areas.
     */
    public void closeAllAreas()
                       throws DMSException;
	
    /**
     * Closes the specified area.
     * @param areaID the identifier used to indicate which area to close.
     * @throws DMSException - thrown if an error occurs while closing the specified area.
     */
    public void closeArea(int areaID)
                   throws DMSException;
	
    /**
     * Closes the specified set of areas.
     * @param areas a set of area identifiers used to indicate which areas to close
     * @throws DMSException - thrown if an error occurs while closing the specified set of areas.
     * @throws IllegalArgumentException  thrown if the areas argument is null.
     */
    public void closeAreas(Set<Integer> areas)
                    throws DMSException;
    
    /**
     * Returns an empty record of the specified type, which is used to store a new record
     * in the database.
     * @param recordType the type of record
     * @return a DMSRecord instance
     * @throws DMSException - thrown if an error occurs while creating a DMSRecord instance
     */
    public DMSRecord createRecord(int recordType)
                           throws DMSException;
    
    /**
     * Gets a DMSArea instance for the specified areaID.
     * @param areaID - an area identifier
     * @return a DMSArea instance
     * @throws DMSException - thrown if an error occurs while getting a DMSArea instance
     */
    public DMSArea getArea(int areaID)
                    throws DMSException;
    
    /**
     * Called to get a DMSDMCA instance.
     * @return DMSDMCA instance
     * @throws DMSException - thrown if an error occurs while getting a DMSDMCA instance
     */
    public DMSDMCA getDMCA() throws DMSException;
    
    /**
     * Gets a DMSIndexSequentialRecordList instance for the specified recordType.
     * @param recordType the type of record
     * @return a DMSIndexSequentialRecordList instance
     * @throws DMSException - thrown if an error occurs while getting a DMSIndexSequentialRecordList
     * instance
     */
    public DMSIndexSequentialRecordList getIndexSequentialRecordList(int recordType)
                                                              throws DMSException;
    
    /**
     * Returns a DMSRecord instance that is on the same CALC chain that has the same database key
     * as the specified DMSRecord instance.  The DMSRecord instance provided as input must have
     * a location mode of CALC.
     * @param record a CALC record instance
     * @return a DMSRecord instance
     * @throws DMSRecordNotFoundException  thrown if the requested record can not be found
     * @throws DMSException - thrown if an error occurs while getting a DMSRecord instance
     * @throws IllegalArgumentException  thrown if the record argument is null
     */
    public DMSRecord getNextDuplicateRecord(DMSRecord record) 
                                     throws DMSRecordNotFoundException, DMSException;
    
    /**
     * Returns a DMSRecord instance associated with the specified database key.
     * @param databaseKey a database key used to locate a record
     * @return a DMSRecord instance
     * @throws DMSRecordNotFoundException  thrown if the requested record can not be found
     * @throws DMSException - thrown if an error occurs while getting a DMSRecord instance
     * @throws IllegalArgumentException  thrown if the databaseKey argument is null
     */
    public DMSRecord getRecord(DMSDatabaseKey databaseKey)
                        throws DMSRecordNotFoundException, DMSException;
    
    /**
     * Returns a DMSRecord instance associated with the specified database key.
     * @param databaseKey a database key used to locate a record
     * @return a DMSRecord instance
     * @throws DMSRecordNotFoundException  thrown if the requested record can not be found
     * @throws DMSException - thrown if an error occurs while getting a DMSRecord instance
     */
    public DMSRecord getRecord(long databaseKey)
                        throws DMSRecordNotFoundException, DMSException;
            
    /**
     * Returns a DMSRecord instance of the specified type, using the supplied key 
     * to locate the record.
     * @param recordType the type of record
     * @param key the ID of the key item
     * @param value the key value
     * @return a DMSRecord instance
     * @throws DMSRecordNotFoundException  thrown if the requested record can not be found
     * @throws DMSException - thrown if an error occurs while getting a DMSRecord instance
     * @throws IllegalArgumentException  thrown if the value argument is null
     */
    public DMSRecord getRecord(int recordType,
                               int key,
                               Object value)
                        throws DMSRecordNotFoundException, DMSException;
    
    /**
     * Returns a DMSRecord instance of the specified recordType, using the supplied keys
     * to locate the record.
     * @param recordType - the type of record
     * @param keys - a Map of key/value pairs used to locate a record
     * @return a DMSRecord instance
     * @throws DMSRecordNotFoundException  thrown if the requested record can not be found
     * @throws DMSException - thrown if an error occurs while getting a DMSRecord instance
     * @throws IllegalArgumentException  thrown if the keys argument is null
     */
    public DMSRecord getRecord(int recordType,
                               Map<Integer, Object> keys)
                        throws DMSRecordNotFoundException, DMSException;
    
    /**
     * Returns a DMSRecord instance of the specified type, using the supplied keys to locate the record.
     * @param recordType type of record
     * @param keys key/value pairs used to locate a record
     * @param charsetNames character set names
     * @return a DMSRecord instance
     * @throws DMSRecordNotFoundException thrown if the requested record can not be found
     * @throws DMSException thrown if an error occurs while getting a DMSRecord instance
     * @throws IllegalArgumentException thrown if the keys argument is null
     */
    public DMSRecord getRecord(int recordType,
                               Map<Integer, Object> keys,
                               Map<Integer, String> charsetNames)
                        throws DMSRecordNotFoundException, DMSException;
    
    /**
     * Opens all areas using the specified usageMode.
     * @param usageMode an identifier that indicates the usageMode used to open all areas.
     * @throws DMSException - thrown if an error occurs while opening an area
     */
    public void openAllAreas(int usageMode)
                      throws DMSException;
    
    /**
     * Opens the specified area with the specified usage mode.
     * @param areaID the identifier of the area to open.
     * @param usageMode indicates the usage mode used when opening the area.
     * @throws DMSException - thrown if an error occurs while opening the specified area
     */
     public void openArea(int areaID,
                          int usageMode)
                   throws DMSException;
     
    /**
     * Opens the specified list of areas.
     * @param areas a Map instance that contains information about which areas to open and
     * the usage mode for each area.  Each key is an areaID and the associated value is the
     * usage mode for the area.
     * @throws DMSException - thrown if an error occurs while opening the list of areas
     * @throws IllegalArgumentException  thrown if the areas argument is null
     */
    public void openAreas(Map<Integer, Integer> areas)
                   throws DMSException;

    /**
     * Sets an access key, which is used to control database operation.
     * @param dataNameID a data name identifier
     * @param accessKey the access key
     * @throws DMSException - thrown if an error occurs while setting an access key
     * @throws IllegalArgumentException  thrown if the accessKey argument is null
     */
    public void setAccessKey(int dataNameID,
                             String accessKey)
                      throws DMSException;

    /**
     * Sets an area key, needed to perform a database operation.
     * @param dataNameID a data name identifier
     * @param pageNumber a page number
     * @param recordNumber a record number
     * @throws DMSException - thrown if an error occurs while setting an area key
     */
    public void setAreaKey(int dataNameID,
                           int pageNumber,
                           int recordNumber)
                    throws DMSException;
    
    /**
     * Sets an area name, needed to perform a database operation.
     * @param dataNameID a data name identifier
     * @param areaName an area name
     * @throws DMSException - thrown if an error occurs while setting an area name
     */
    public void setAreaName(int dataNameID,
                            String areaName)
                     throws DMSException;
    
    /**
     * Called to set the character set used for decoding and encoding of String values.
     * @param characterSet a character set name
     * @throws DMSException thrown if an unsupported character set name is specified
     * @throws IllegalArgumentException thrown if the characterSet argument is null
     */
    public void setCharacterSet(String characterSet)
                         throws DMSException;
    
    /**
     * Called to set the FetchWithLock value.  The FetchWithLock feature enables exclusive 
     * paging locking when getting records.  This feature helps to prevent certain deadlock 
     * situations.
     * <p>
     * Note:  This method can be called only if the FetchWithLock value in effect when 
     * creating a DMSDatabaseManager instance is DYNAMIC_FETCH_WITH_LOCK.
     * @param fetchWithLock the value NO_FETCH_WITH_LOCK or DYNAMIC_FETCH_WITH_LOCK.
     * @throws DMSException thrown if an error occurs while setting the fetchWithLock value
     */
    public void setFetchWithLock(int fetchWithLock)
                          throws DMSException;
    
    /**
     * Called to identify the fill character used to pad String item values when performing 
     * database operations.
     * <p>
     * This capability is useful when the fill character for a String item value in the 
     * database is defined as some character other then a space, such as binary zero, or 
     * LOW-VALUE in COBOL terms.  The correct fill character must be used to locate records 
     * in the database when one or more keys are defined as String items.  The fill character 
     * is also used to pad String item values with the correct fill character when modifying 
     * and storing a record in the database.
     * @param fillCharacter the identity of the character used to pad String item values when 
     *                      performing database operations.
     *                      Valid values are "SPACE", "LOW_VALUE", and "HIGH_VALUE".

     * @throws DMSException  thrown if an error occurs while setting the fillCharacter value.
     */
    public void setFillCharacter(String fillCharacter) 
                          throws DMSException;
    
    /**
     * Sets the maximum page number, needed to perform a database operation.
     * @param dataNameID a data name identifier
     * @param pageNumber a page number
     * @throws DMSException - thrown if an error occurs while setting the maximum page number
     */
    public void setMaxPage(int dataNameID,
                           int pageNumber)
                    throws DMSException;

    /**
     * Sets the minimum page number, needed to perform a database operation.
     * @param dataNameID a data name identifier
     * @param pageNumber a page number
     * @throws DMSException - thrown if an error occurs while setting the minimum page number
     */
    public void setMinPage(int dataNameID,
                           int pageNumber)
                    throws DMSException;
    
    /**
     * Called to set the priority used to determine rollback priority when database deadlock 
     * occurs.  The program with the lowest priority is selected for rollback processing.
     * @param priority defines rollback priority for the program
     * @throws DMSException  thrown if an error occurs while setting the priority.
     */
    public void setPriority(int priority)
                     throws DMSException;

}
