/*
* Module:       com.unisys.os2200.dms.DMSDatabaseKeyList.java
*
* Description:  The DMSDatabaseKeyList class extends the java.util.ArrayList class.  This class 
*               is used to return a collection of database keys from a pointer array or indexed 
*               pointer array recordSet.
* 
* Product:      Resource Adapter for DMS
*
* Date:         April, 2006
*
* Copyright (c) 2010 Unisys Corporation.
*
* All rights reserved.
* 
* UNISYS CONFIDENTIAL
*/
package com.unisys.os2200.dms;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

/**
 * The DMSDatabaseKeyList class extends the java.util.ArrayList class.  This class is used to 
 * return a collection of database keys from a pointer array or indexed pointer array recordSet.
 */
public final class DMSDatabaseKeyList extends ArrayList<DMSDatabaseKey> {

    // Universal version identifier for a Serializable class
    static final long serialVersionUID = -6777351711063175954L;

    /**
     * Class Constructor with no arguments.
     */
    public DMSDatabaseKeyList() {
        super();
    }

    /**
     * Class Constructor with an initial capacity argument.
     * 
     * @param initialCapacity the initial capacity of the database key list
     */
    public DMSDatabaseKeyList(int initialCapacity) {
        super(initialCapacity);
    }

    /**
     * Class Constructor with a collection argument.
     * 
     * @param collection a collection of database keys used to initialize this list
     */
    public DMSDatabaseKeyList(Collection<DMSDatabaseKey> collection) {
        super(collection);
    }

    /**
     * Called to create a new database key list that contains database keys that 
     * are common to both lists.
     * 
     * @param databaseKeyList a DMSDatabaseKeyList instance 
     * @return a DMSDatabaseKeyList instance
     */
    public DMSDatabaseKeyList intersect(DMSDatabaseKeyList databaseKeyList) {
        /*
         * The implementation of this method will iterate through the specified databaseKeyList 
         * and for each element, call the this.contains() method to determine if the database 
         * key exists in both lists.  Database keys that exist in both lists will be added to 
         * the new DMSDatabaseKeyList instance.
         */
        DMSDatabaseKeyList databaseKeyListIntersect = new DMSDatabaseKeyList();
        
        Iterator<DMSDatabaseKey> it = databaseKeyList.iterator();
        
        DMSDatabaseKey databaseKey = null;
        
        while ( it.hasNext() ) {
            databaseKey = (DMSDatabaseKey)it.next();
            
            if ( this.contains(databaseKey) ) {
                databaseKeyListIntersect.add(databaseKey);
            }
        }
        return databaseKeyListIntersect;
    }
    
    /**
     * Called to create a new database key list that contains unique database keys 
     * from both lists.
     * 
     * @param databaseKeyList a DMSDatabaseKeyList instance
     * @return a DMSDatabaseKeyList instance
     */
    public DMSDatabaseKeyList union(DMSDatabaseKeyList databaseKeyList) {
        /*
         * The implementation constructs a new DMSDatabaseKeyList instance, which is 
         * initialize with the database keys from this list.  Then, iterate through 
         * the specified databaseKeyList and for each element, call the this.contains() 
         * method to determine if the database key is unique.  Database keys that are 
         * unique will be added to the new DMSDatabaseKeyList instance.
         */
        DMSDatabaseKeyList databaseKeyListUnion = new DMSDatabaseKeyList(this);
        
        Iterator<DMSDatabaseKey> it = databaseKeyList.iterator();
        
        DMSDatabaseKey databaseKey = null;
        
        while ( it.hasNext() ) {
            databaseKey = (DMSDatabaseKey)it.next();
            
            if ( !this.contains(databaseKey) ) {
                databaseKeyListUnion.add(databaseKey);
            }
        }
        return databaseKeyListUnion;
    }
    
    /**
     * Returns true if this list contains the specified databaseKey.
     * 
     * @param databaseKey a DMSDatabaseKey instance
     * @return true if the specified databaseKey is present; false otherwise.
     */
    public boolean contains(DMSDatabaseKey databaseKey) {
        
        boolean result = false;
        
        if ( databaseKey != null ) {
            for (int i = 0; i < this.size(); i++) {
                if ( databaseKey.equals( (DMSDatabaseKey)(this.get(i)) ) ) {
                    result = true;
                    break;
                }
            }
        }
        
        return result;
    }
}
