/*
* Module:       com.unisys.os2200.dms.DMSConnection.java
*
* Description:  Interface which defines the methods used to establish a connection to a DMS database.
*               The DMSConnection is an application-level handle used to isolate an application component
*               from the physical connection to the DMS system.
*               
* Product:      Resource Adapter for DMS
*
* Date:         April, 2006
*
* Copyright (c) 2009 Unisys Corporation.
*
* All rights reserved.
*
* UNISYS CONFIDENTIAL
*/

package com.unisys.os2200.dms;

import java.util.Map;
import javax.resource.spi.LocalTransaction;

/**
 * The DMSConnection interface defines the methods used to access DMS data.  The DMSConnection is
 * an application-level handle used to isolate an application component from the physical connection
 * to the DMS system.
 */
public interface DMSConnection {

    /**
     * Indicates that an INVOKE key is needed to access a subschema.
     * Value is the string value defined in the subschema for the INVOKE key.
     * <p>
     * Value Object: java.lang.String
     */
    static final int DMS_INVOKE_KEY = 1;    
    /**
     * Indicates that an IMPART key is needed to connect to a subschema.
     * Value is the string value defined in the schema for the IMPART key.
     * <p>
     * Value Object: java.lang.String
     */
    static final int DMS_IMPART_KEY = 2;    
    /**
     * Indicates if exclusive page locking is enabled when getting records.
     */ 
    static final int DMS_FETCH_WITH_LOCK = 3;    
    /**
     * Indicates if queuing is enabled when a page or area is not
     * immediately available.
     */ 
    static final int DMS_QUEUING = 4;

    /* Fetch With Lock Settings */

    /**
     * Disables exclusive page lock when getting a record.
     */
    static final int NO_FETCH_WITH_LOCK = 0;
    /**
     * Enables exclusive page lock when getting a record.  
     * This setting remains in effect until the DMSDatabaseManager 
     * instance is closed.
     */
    static final int STATIC_FETCH_WITH_LOCK = 1;
    /**
     * Enables exclusive page lock when getting a record.
     * This setting can be changed during the life of a 
     * DMSDatabaseManager instance.
     */
    static final int DYNAMIC_FETCH_WITH_LOCK = 2;

    /* No-Queue Settings */

    /**
     * Enables request queuing when a page or area is not immediately
     * available.
     */
    static final int QUEUING_ENABLED = 0;
    /**
     * Returns an error when a page or area is not immediately
     * available.  
     */    
    static final int QUEUING_DISABLED = 1;
    
    /**
     * Returns a DMSDatabaseManager instance associated with the specified database name.
     * @param databaseName name of the DMS database to access.
     * @throws DMSException - thrown if an error occurs while getting a DMSDatabaseManager.
     * @throws IllegalArgumentException  thrown if the databaseName argument is null.
     */
    public DMSDatabaseManager getDatabaseManager(String databaseName)
                                          throws DMSException;

    /**
     * Returns a DMSDatabaseManager instance associated with the specified database name.
     * @param databaseName name of the DMS database to access.
     * @param configuration a Map instance that contains configuration information needed to
     *                      acquire a DMSDatabaseManager instance.
     * @return a DMSDatabaseManager instance.
     * @throws DMSException  thrown if an error occurs while getting a DMSDatabaseManager instance.
     * @throws IllegalArgumentException - thrown if the databaseName argument is null.
     */
    public DMSDatabaseManager getDatabaseManager(String databaseName,
                                                 Map<Integer, Object> configuration)
                                          throws DMSException;
    
    /** 
     * Returns a LocalTransaction instance that enables a component to demarcate resource manager
     * local transactions on the Connection.
     * @throws DMSException - thrown if an error occurs while getting a LocalTransaction instance.
     */
    public LocalTransaction getLocalTransaction()
                                         throws DMSException;

    /**
     * Closes the application-level handle used to access a DMS system.
     * @throws DMSException - thrown if an error occurs while closing access to the DMS system.
     */
    public void close()
               throws DMSException;
    
}
